/*
This provides a single mechanism for loading js files.
It is invoked with a list of js file to load. The list contains the base names of the js files.
There may be a need to load a corresponding css file for a given js file, which is handled here as well.

Supported forms of the URL to this file that indicate the files to load:
   .../loader.js?collection,cookie,datepicker  --> load collection.js and cookie.js and datepicker.js and datepicker.css
   .../loader.js?queue,cookie,vincore          --> load queue.js and cookie.js and vincore.js and vincore.css

There is historical record for allowing URL to be in this form, but when it is used, it does not do anything anymore.
   .../loader.js?handlerPath=abc

The basic form is used in the VIN source code.
As the generated HTML flows out through F5, an irule exists to inject a cache busting parameter named "random".
This does not happen on localhost.
This causes the browser to get a fresh copy of loader.js file when a new iteration is deployed that changes the random value.
The result is this file sees itself called with another paramter in the	query string.
 .../loader.js?random=20211118&collection,cookie,datepicker 
  --> load collection.js?random=20211118 and cookie.js?random=20211118 and datepicker.js?random=20211118 and datepicker.css?random=20211118

Currently built around Prototype, Scriptaculous, and my (Randal) own extensions to said libraries.
2009.10.30 > Firebug Lite added
2009.11.04 > HtmlToolbar added
2010.03.29 > Moved libJai code to prototype, except Collection (renamed file); removed multi-core approach
2011.01.17 > (summarizing additions i forgot to mark)
			WYSIWYG, CheckBoxDropDown, Dictionary, FindReplace added.
			HtmlToolbar removed. (replaced by WYSIWYG)
			CSS link inclusion added.
			Ability for Loader to manage non-prototype script includes to ensure single load.
2011.04.04 > Flash added (used for detecting flash version)
			Prototype now relies on flash.js to set Prototype.BrowserFeatures.Flash if flash is enabled and new enough to work on VIN
2011.05.01 > Update to Prototype 1.7 complete (haven't looked at Scriptaculous changes)
			Modifications to Prototype are now at the bottom of prototype.js, no longer integrated into the body of the code
2011.10.18 > Modified to be used with BaseHandler for SCRIPT / STYLE grouping
2021.11.18 > Added code to handle receiving a possible cache-busting parameter by way of an F5 irule. 
               Pass this parameter along in the URLs used to fetch the loaded js files
			 Remove code to load files by way of a handler endpoint.  This feature had been dropped earlier but the code was not removed.
			 Remove code to handle alternate forms for listing the files to load.  
			    No more usage like "loader.js?load=abc,xyz" or "loader.js?ensure=abc,xyz".
*/
if (!window.JS_Lib_Loader)
	window.JS_Lib_Loader = {
	All         : [],
	Requires    : [],
	CSS         : [],
	Loaded      : [],
	Ensured     : [],
	HandlerPath : null,
	_path       : null,
	CacheBuster : "",  // '?random=123' or ''
	add: function(name, reqs, css) {
		this.All.push(name);
		this.Requires.push(reqs);
		this.CSS.push(css)
		this.Loaded.push(false);
	},
	all: function(bool) {
		for (var i = 0, l = this.All.length; i < l; i++)
			if (this.Loaded[i] == !true)
				return false;
		return true;
	},
	ensure: function(src) {
		this.Ensured.push(src);
		document.write("<scr" + "ipt type='text/javascript' src='" + src + this.CacheBuster + "'></scr" + "ipt>");
	},
	init: function () {

	    if (this.All.length) return;

		// For each js file base name, register other js files to load, and if the js file has a corresponding css file to load.
		//        libname             required libs                        hasCSS
		this.add("flash",             false,                               false); // Randal
		this.add("rangy",             false,                               false);
		this.add("foresight",         false,                               false); // https://github.com/adamdbradley/foresight.js (Hi-Res image switching)
        this.add("prototype", ["flash", "rangy"], false); // Prototype; doesn't require anything, i just use it to force some
		this.add("prototype_ext",   ["primitives","sel","color"],          false); // Randal; All the mods/addons for Prototype's core

		this.add("builder",          "prototype",                          false); // Scriptaculous
		this.add("effects",          "prototype",                          false); // Scriptaculous
		this.add("slider",           "prototype",                          false); // Scriptaculous
		this.add("sound",            "prototype",                          false); // Scriptaculous
		this.add("controls",         "effects",                            false); // Scriptaculous
		this.add("dragdrop",         "effects",                            false); // Scriptaculous

		this.add("primitives",       "prototype",                          false); // Randal; Enum, int, bool, float, DateTime
		this.add("sel",              "prototype",                          false); // Randal; Manages and manipulates ranges and selections
		this.add("color",            "prototype",                          false); // Randal; HTML color names, hex values, and conversions
		this.add("uploader",         "prototype",                          false); // Randal
		this.add("vincore",          "prototype_ext",                      true);  // Randal; ToolTipHandler makes use of Effects 'if' loaded

		this.add("autocomplete",     "vincore",                            true);  // Randal
		this.add("checkboxdropdown", "vincore",                            true);  // Randal
		this.add("collection",       "vincore",                            false); // Randal
		this.add("cookie",           "vincore",                            false); // Randal
		this.add("datepicker",       "vincore",                            true);  // Randal
		this.add("h5db",            ["vincore"],                           false); // Randal
		this.add("queue",           ["vincore"],                           false); // Randal
		this.add("tablekit",         "vincore",                            false); // Russell

		this.add("accordion",       ["vincore","effects"],                 false); // Russell

		this.add("splitpane",        "dragdrop",                           false); // Randal
		this.add("tablehandler",     "dragdrop",                           false); // Randal

		this.add("menus",           "collection",                          false); // Randal
		this.add("toolbox",         ["collection","dragdrop"],             true);  // Randal

		this.add("dialogs",         ["queue","toolbox","uploader"],        true);  // Randal

		this.add("toolbarcontext",  ["queue","vincore"],                   false); // Randal
		this.add("dictionary",      ["toolbox","toolbarcontext"],          true);  // Randal
		this.add("findreplace",     ["toolbox","toolbarcontext"],          true);  // Randal
		this.add("wysiwyg",         ["dictionary","findreplace","menus"],  true);  // Randal
	},
	require: function(index, hash) {
		var lib = this.All[index] + ".js";  // name used to check if it is to be skipped
		var jsFileName = this.All[index] + ".js" + this.CacheBuster;
		var cssFileName = this.All[index] + ".css" + this.CacheBuster;
		this.Loaded[index] = true;
        
	    // To allow for runtime skipping of the default javascript files,
		// place a tag like this immediately under the page's doctype declaration: 
		//   <NOLOADJS skip="sel.js,rangy.js"></NOLOADJS>
		// The tag should have the full file names to skip in the "skip" attribute. This next bit of logic will cause them to be skipped.
		var noLoad = document.getElementsByTagName("NOLOADJS");

		if (noLoad.length > 0)
		{
		    var skip = noLoad[0].getAttribute('skip').split(",");
		    
		    if (skip.indexOf(lib) >= 0)
		        return;
		}

		document.write("<scr" + "ipt type='text/javascript' src='" + this._path + jsFileName + (hash ? "#" + hash : "") + "'></scr" + "ipt>");
		if (this.CSS[index])
			document.write("<link type='text/css' rel='stylesheet' href='" + this._path + "css/" + cssFileName + (hash ? "#" + hash : "") + "'/>");
	},
	reqs: function(item) {
		var L = window.JS_Lib_Loader, ret = [], req = false;
		if (-1 < L.All.indexOf(item)) req = L.Requires[L.All.indexOf(item)];
		if (req) {
			if (typeof req == "object" && 'splice' in req && 'join' in req) {
				for (var i = 0, l = req.length; i < l; i++)
					ret.push(L.reqs(req[i]));
			}else {
				ret.push(L.reqs(req));
			}
			ret.push(req);
		};
		return ret;
	},
	load: function() {
		//document.domain = -1 < location.host.toLowerCase().indexOf("vin.com") ? "vin.com" : location.host;
		this.init();
		if (typeof(Array.prototype.indexOf) == "undefined")
			Array.prototype.indexOf = function(item) { for (var i = 0, len = this.length; i < len; i++) if (this[i] == item) return i; return -1; };

		var L = window.JS_Lib_Loader;
		if (L.all()) return;

		var oldIE = -1 < window.navigator.userAgent.indexOf("MSIE") && parseInt(window.navigator.userAgent.split("MSIE")[1].split(';')[0], 10) < 9;

		//Loop through all SCRIPT tags to find one that has src= value that is us, loader.js
		var scripts = document.getElementsByTagName("SCRIPT");
		for (var z = 0, len = scripts.length; z < len; z++) {
			var s = scripts[z].src.toLowerCase().split('#')[0];
			var hash = scripts[z].src.split('#')[1] || "";

			//We only work with loader.js SCRIPTs 
			if (s.indexOf("/loader.js?") < 0) continue;

		    //  ignore case of this being included with the 'handlerpath=abc' parameter.
			//    ../loader.js?handlerpath=abc   -->F5.irule--> ../loader.js?random=123&handlerpath=abc
			//  that is an old feature that would set the HandlerPath endpoint to be used to get js/css files
			//     by referencing an endpoint handler instead of getting them directly.
			//  use cases of this script tag exist within VIN code but there is no longer code here to support that feature.
			//  those use cases to request using an endpoint are ignored, and all js/css files are gotten directly.
			//  by skipping it, the default value of HandlerPath remains, and nothing special is done here.
			if (s.split('/loader.js?')[1].indexOf("handlerpath=") >= 0) {
				//L.HandlerPath = s.split('handlerpath=')[1] || null;
				continue;
			}

			// get the path to this js file.  js files to load are expected to be in this same folder.
			L._path = s.replace(/loader\.js?.*/i, '');

			// Get the possible cache-busting parameter "random" and the comma-seperated list of file names to load.
			// The random=nnn is not injected into the request when not going through F5, for instance on localhost.
			// ../loader.js?queue,cookie,vincore  -->F5.irule--> ../loader.js?random=123&queue,cookie,vincore
			var includes = "";  // this will hold the list of names of file to include
			if (s.split('/loader.js?')[1].indexOf("random=") == 0) {
				// random=nnn parameter exists.
				// we have: ../loader.js?random=123&queue,cookie,vincore
				var getRightOfRandom = s.split('random=');  // [../loader.js?] [123&abc,xyz]
			    var ror = 1 < getRightOfRandom.length ? getRightOfRandom[1] : "";  // '123&abc,xyz' or ''
				var rorsplit = ror.split('&');  //   [123] [abc,xyz]
				var randomValue =  rorsplit[0]; // [123]
				this.CacheBuster = '?random=' + randomValue;
				includes = 1 < rorsplit.length ? rorsplit[1] : ""; // 
			}
			else {
				// no random=nnn parameter.
				// we have: ../loader.js?queue,cookie,vincore
			    var getQueryString = s.split("?");
			    includes = 1 < getQueryString.length ? getQueryString[1] : "";
			}


			var inc = [], reqs;
			includes = includes.split(','); // create array of requested names
			// Always include VINcore
			includes.push("vincore");

			// For each requested name, add in dependent required names that they need to have
			for (var i = 0, ilen = includes.length; i < ilen; i++) {
				if (!!includes[i] && inc.indexOf(includes[i]) < 0) {
					reqs = L.reqs(includes[i]);
					for (var j = 0, jlen = reqs.length; j < jlen; j++)
						inc.push(reqs[j]);
					inc.push(includes[i]);
				}
			}

			//Flatten and remove duplicates
			includes = inc.toString().split(','); inc = [];
			for (var i = 0, ilen = includes.length; i < ilen; i++)
				if (includes[i] && inc.indexOf(includes[i]) < 0)
					inc.push(includes[i]);

			//  Loop through and render out the SCRIPT tags with direct reference to the files.
			//  L.require() is function that emits the direct include, with cache buster param added to URL
			//  L.ensure() is to emit a direct include of a file that is not one of the standard ones known about in the init().
			if (inc.length) {
				for (var i = 0, ilen = inc.length; i < ilen; i++) {
					var incIndex = L.All.indexOf(inc[i]);
					if (incIndex < 0) {
						// Case where the requested js file is not one of the registered ones.
						// It must be added to the query string list with the full name including the .js ending.
						// There is no model for emitting a related css file.
						// There is not a known case of loader.js getting called with an unregistered file.
						if (L.Ensured.indexOf(inc[i]) < 0)
							L.ensure(inc[i]);
					}else {
						if (!L.Loaded[incIndex]) {
							L.require(incIndex, hash);
						}
					}
				}
			}

		}
	}
};
window.JS_Lib_Loader.load();